import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { 
  Building2, 
  Users, 
  Award, 
  Clock, 
  CheckCircle, 
  Phone, 
  Mail,
  MapPin,
  Star,
  Shield,
  Zap,
  Wrench,
  Globe
} from "lucide-react";
import Link from "next/link";
import Image from "next/image";
import { 
  getCompanyInfo, 
  getAllProducts, 
  getHomePageFeatures, 
  getAllTestimonials,
  getDataWithFallback,
  getImageUrl,
  fallbackData,
  CompanyInfo,
  Product,
  Feature,
  Testimonial
} from "@/lib/contentful";

// Default/fallback data for when CMS is not configured
const defaultCompanyInfo: CompanyInfo = {
  companyName: "SAAM Cabins",
  tagline: "Premium Porta Cabins & Containers in UAE",
  description: "Leading manufacturer and supplier of high-quality porta cabins, security cabins, and containers across the United Arab Emirates.",
  phoneNumber: "+971 58 201 2073",
  whatsappNumber: "+971 58 201 2073", 
  email: "info@saamcabins.com",
  address: "Industrial Area, Sharjah, UAE",
  workingHours: "Sunday - Thursday: 8:00 AM - 6:00 PM, Saturday: 9:00 AM - 5:00 PM"
};

const defaultFeatures: Feature[] = [
  {
    title: "Premium Quality",
    description: "ISO certified manufacturing with premium materials and construction standards",
    icon: "Award",
    order: 1,
    isActive: true
  },
  {
    title: "Fast Delivery", 
    description: "Quick manufacturing and delivery across all UAE emirates",
    icon: "Zap",
    order: 2,
    isActive: true
  },
  {
    title: "Custom Solutions",
    description: "Tailored designs to meet specific requirements and applications",
    icon: "Wrench", 
    order: 3,
    isActive: true
  },
  {
    title: "Professional Service",
    description: "Expert consultation, installation, and after-sales support",
    icon: "Users",
    order: 4,
    isActive: true
  },
  {
    title: "Nationwide Coverage",
    description: "Service coverage across Dubai, Abu Dhabi, Sharjah, and all UAE",
    icon: "Globe",
    order: 5,
    isActive: true
  },
  {
    title: "Security & Safety",
    description: "Enhanced security features and safety compliance standards",
    icon: "Shield",
    order: 6,
    isActive: true
  }
];

const defaultProducts: Product[] = [
  {
    name: "Security Cabins",
    slug: "security-cabin",
    category: "Security",
    shortDescription: "Professional security cabins for monitoring and surveillance applications",
    fullDescription: {} as any,
    features: ["Bullet-proof glass options", "Climate control systems", "Communication equipment", "CCTV integration"],
    specifications: [],
    applications: ["Construction sites", "Industrial facilities", "Government buildings", "Border checkpoints"],
    images: [],
    isActive: true,
    order: 1
  },
  {
    name: "Office Cabins", 
    slug: "office-cabin",
    category: "Office",
    shortDescription: "Modern office solutions for temporary and permanent installations",
    fullDescription: {} as any,
    features: ["Professional interiors", "IT infrastructure ready", "Meeting room configurations", "Ergonomic design"],
    specifications: [],
    applications: ["Construction site offices", "Remote work locations", "Business expansion", "Temporary facilities"],
    images: [],
    isActive: true,
    order: 2
  },
  {
    name: "Toilet Units",
    slug: "toilet-units", 
    category: "Sanitation",
    shortDescription: "Hygienic and durable toilet and washroom facilities",
    fullDescription: {} as any,
    features: ["Water-efficient fixtures", "Ventilation systems", "Easy maintenance", "Disabled access options"],
    specifications: [],
    applications: ["Construction sites", "Events and festivals", "Public facilities", "Emergency relief"],
    images: [],
    isActive: true,
    order: 3
  }
];

const iconMap = {
  Award: Award,
  Zap: Zap,
  Wrench: Wrench,
  Users: Users,
  Globe: Globe,
  Shield: Shield,
  Building2: Building2,
  Clock: Clock,
  CheckCircle: CheckCircle
};

export default async function HomePage() {
  // Fetch CMS data with fallbacks
  const companyInfo = await getDataWithFallback(getCompanyInfo, defaultCompanyInfo);
  const features = await getDataWithFallback(getHomePageFeatures, defaultFeatures);
  const products = await getDataWithFallback(getAllProducts, defaultProducts);
  const testimonials = await getDataWithFallback(getAllTestimonials, []);

  const IconComponent = ({ iconName }: { iconName: string }) => {
    const Icon = iconMap[iconName as keyof typeof iconMap] || Building2;
    return <Icon className="h-8 w-8" />;
  };

  return (
    <div className="min-h-screen">
      {/* Hero Section - CMS Driven */}
      <section className="relative py-20 bg-gradient-to-r from-blue-900/90 to-slate-900/90">
        <div className="absolute inset-0 bg-black/40"></div>
        {companyInfo?.heroImage && (
          <div className="absolute inset-0">
            <Image
              src={getImageUrl(companyInfo.heroImage, 1920, 1080, 80)}
              alt="SAAM Cabins"
              fill
              className="object-cover"
              priority
            />
          </div>
        )}
        {!companyInfo?.heroImage && (
          <div className="absolute inset-0">
            <div className="w-full h-full bg-gradient-to-br from-blue-800 via-slate-800 to-gray-800"></div>
          </div>
        )}
        
        <div className="relative z-20 container mx-auto px-4 text-center text-white">
          <h1 className="text-4xl md:text-6xl font-bold mb-6">
            {companyInfo?.tagline || "PORTA CABIN SOLUTIONS!"}
          </h1>
          <p className="text-xl md:text-2xl mb-8 max-w-3xl mx-auto">
            {companyInfo?.description}
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="bg-blue-600 hover:bg-blue-700 text-white px-8">
              <Phone className="mr-2 h-5 w-5" />
              Call: {companyInfo?.phoneNumber}
            </Button>
            <Button size="lg" variant="outline" className="border-white text-white hover:bg-white/10 px-8" asChild>
              <Link href="/contact">
                <Mail className="mr-2 h-5 w-5" />
                Get Quote
              </Link>
            </Button>
          </div>
        </div>
      </section>

      {/* Company Stats */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-8 text-center">
            <div>
              <h3 className="text-3xl font-bold text-blue-600 mb-2">500+</h3>
              <p className="text-gray-600">Projects Completed</p>
            </div>
            <div>
              <h3 className="text-3xl font-bold text-blue-600 mb-2">50+</h3>
              <p className="text-gray-600">Happy Clients</p>
            </div>
            <div>
              <h3 className="text-3xl font-bold text-blue-600 mb-2">10+</h3>
              <p className="text-gray-600">Years Experience</p>
            </div>
            <div>
              <h3 className="text-3xl font-bold text-blue-600 mb-2">24/7</h3>
              <p className="text-gray-600">Support Available</p>
            </div>
          </div>
        </div>
      </section>

      {/* Features Section - CMS Driven */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-6">
              Why Choose {companyInfo?.companyName}?
            </h2>
            <p className="text-lg text-gray-700 max-w-3xl mx-auto">
              We provide comprehensive porta cabin solutions with unmatched quality and service across the UAE.
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {features.map((feature, index) => (
              <Card key={index} className="text-center hover:shadow-lg transition-shadow">
                <CardHeader>
                  <div className="bg-blue-100 w-16 h-16 rounded-full flex items-center justify-center mx-auto mb-4">
                    <div className="text-blue-600">
                      <IconComponent iconName={feature.icon} />
                    </div>
                  </div>
                  <CardTitle className="text-xl font-bold">{feature.title}</CardTitle>
                  <CardDescription>{feature.description}</CardDescription>
                </CardHeader>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Products Showcase - CMS Driven */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-6">
              Our Product Range
            </h2>
            <p className="text-lg text-gray-700 max-w-3xl mx-auto">
              Explore our comprehensive range of porta cabins designed for various applications and industries.
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mb-12">
            {products.slice(0, 3).map((product, index) => (
              <Card key={index} className="overflow-hidden hover:shadow-lg transition-shadow">
                <div className="relative h-48 bg-gradient-to-br from-blue-800 to-slate-800">
                  {product.images && product.images.length > 0 ? (
                    <Image
                      src={getImageUrl(product.images[0], 400, 200, 80)}
                      alt={product.name}
                      fill
                      className="object-cover"
                    />
                  ) : (
                    <div className="absolute inset-0 flex items-center justify-center">
                      <Building2 className="h-16 w-16 text-white/50" />
                    </div>
                  )}
                  <Badge className="absolute top-4 left-4 bg-blue-600">
                    {product.category}
                  </Badge>
                </div>
                
                <CardHeader>
                  <CardTitle className="text-xl font-bold">{product.name}</CardTitle>
                  <CardDescription>{product.shortDescription}</CardDescription>
                </CardHeader>
                
                <CardContent>
                  <div className="space-y-2 mb-4">
                    {product.features.slice(0, 3).map((feature, idx) => (
                      <div key={idx} className="flex items-center text-sm text-gray-600">
                        <CheckCircle className="h-3 w-3 mr-2 text-blue-600" />
                        <span>{feature}</span>
                      </div>
                    ))}
                  </div>
                  
                  <Button className="w-full bg-blue-600 hover:bg-blue-700" asChild>
                    <Link href={`/products/${product.slug}`}>
                      Learn More
                    </Link>
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>

          <div className="text-center">
            <Button size="lg" variant="outline" className="border-blue-600 text-blue-600 hover:bg-blue-50" asChild>
              <Link href="/products">View All Products</Link>
            </Button>
          </div>
        </div>
      </section>

      {/* Testimonials Section - CMS Driven */}
      {testimonials.length > 0 && (
        <section className="py-16 bg-gray-50">
          <div className="container mx-auto px-4">
            <div className="text-center mb-16">
              <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-6">
                Client Testimonials
              </h2>
              <p className="text-lg text-gray-700">
                What our satisfied customers say about our porta cabin solutions
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
              {testimonials.slice(0, 3).map((testimonial, index) => (
                <Card key={index} className="text-center">
                  <CardHeader>
                    <div className="flex justify-center mb-4">
                      {testimonial.avatar ? (
                        <Image
                          src={getImageUrl(testimonial.avatar, 80, 80, 80)}
                          alt={testimonial.customerName}
                          width={80}
                          height={80}
                          className="rounded-full"
                        />
                      ) : (
                        <div className="w-20 h-20 bg-blue-100 rounded-full flex items-center justify-center">
                          <Users className="h-8 w-8 text-blue-600" />
                        </div>
                      )}
                    </div>
                    <div className="flex justify-center mb-4">
                      {[...Array(5)].map((_, i) => (
                        <Star
                          key={i}
                          className={`h-4 w-4 ${
                            i < testimonial.rating ? 'text-yellow-400 fill-current' : 'text-gray-300'
                          }`}
                        />
                      ))}
                    </div>
                  </CardHeader>
                  <CardContent>
                    <p className="text-gray-700 mb-4 italic">"{testimonial.content}"</p>
                    <div>
                      <p className="font-semibold text-gray-900">{testimonial.customerName}</p>
                      <p className="text-sm text-gray-600">{testimonial.position}</p>
                      <p className="text-sm text-gray-600">{testimonial.company}</p>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </section>
      )}

      {/* Call to Action */}
      <section className="py-16 bg-blue-800 text-white">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Ready to Get Started?
          </h2>
          <p className="text-xl mb-8 max-w-2xl mx-auto">
            Contact us today for a free consultation and quote for your porta cabin requirements.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="bg-white text-blue-800 hover:bg-gray-100 px-8">
              <Phone className="mr-2 h-5 w-5" />
              Call: {companyInfo?.phoneNumber}
            </Button>
            <Button size="lg" variant="outline" className="border-white text-white hover:bg-white/10 px-8" asChild>
              <Link href="/contact">
                <Mail className="mr-2 h-5 w-5" />
                Get Free Quote
              </Link>
            </Button>
          </div>
        </div>
      </section>
    </div>
  );
}